<?php
require_once 'auth.php';
require_once 'config.php';
require_once 'ExamManager.php';

if (!Auth::isLoggedIn() || !Auth::isAdmin()) {
    header('Location: admin_login.php');
    exit;
}

$db = Config::getDB();
$action = $_GET['action'] ?? 'list';
$exam_id = $_GET['exam_id'] ?? 0;
$question_id = $_GET['question_id'] ?? 0;

// Handle form submissions
if ($_POST['action'] ?? '' === 'add_exam') {
    $name = Config::sanitize($_POST['name']);
    $total_questions = intval($_POST['total_questions']);
    $duration_seconds = intval($_POST['duration_seconds']) * 60; // Convert to seconds
    $points_per_correct = floatval($_POST['points_per_correct']);
    
    $stmt = $db->prepare("INSERT INTO exams (name, total_questions, duration_seconds, points_per_correct) VALUES (?, ?, ?, ?)");
    if ($stmt->execute([$name, $total_questions, $duration_seconds, $points_per_correct])) {
        $success = "Exam created successfully!";
    } else {
        $error = "Failed to create exam.";
    }
}

if ($_POST['action'] ?? '' === 'edit_exam') {
    $name = Config::sanitize($_POST['name']);
    $total_questions = intval($_POST['total_questions']);
    $duration_seconds = intval($_POST['duration_seconds']) * 60;
    $points_per_correct = floatval($_POST['points_per_correct']);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    $stmt = $db->prepare("UPDATE exams SET name = ?, total_questions = ?, duration_seconds = ?, points_per_correct = ?, is_active = ? WHERE id = ?");
    if ($stmt->execute([$name, $total_questions, $duration_seconds, $points_per_correct, $is_active, $exam_id])) {
        $success = "Exam updated successfully!";
    } else {
        $error = "Failed to update exam.";
    }
}

if ($_POST['action'] ?? '' === 'add_question') {
    $exam_id = intval($_POST['exam_id']);
    $question_text = Config::sanitize($_POST['question_text']);
    $question_type = Config::sanitize($_POST['question_type']);
    $option_a = Config::sanitize($_POST['option_a'] ?? '');
    $option_b = Config::sanitize($_POST['option_b'] ?? '');
    $option_c = Config::sanitize($_POST['option_c'] ?? '');
    $option_d = Config::sanitize($_POST['option_d'] ?? '');
    $correct_answer = Config::sanitize($_POST['correct_answer']);
    
    $stmt = $db->prepare("INSERT INTO questions (exam_id, question_text, question_type, option_a, option_b, option_c, option_d, correct_answer) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    if ($stmt->execute([$exam_id, $question_text, $question_type, $option_a, $option_b, $option_c, $option_d, $correct_answer])) {
        $success = "Question added successfully!";
        $action = 'view_questions';
    } else {
        $error = "Failed to add question.";
    }
}

if ($_POST['action'] ?? '' === 'edit_question') {
    $question_text = Config::sanitize($_POST['question_text']);
    $question_type = Config::sanitize($_POST['question_type']);
    $option_a = Config::sanitize($_POST['option_a'] ?? '');
    $option_b = Config::sanitize($_POST['option_b'] ?? '');
    $option_c = Config::sanitize($_POST['option_c'] ?? '');
    $option_d = Config::sanitize($_POST['option_d'] ?? '');
    $correct_answer = Config::sanitize($_POST['correct_answer']);
    
    $stmt = $db->prepare("UPDATE questions SET question_text = ?, question_type = ?, option_a = ?, option_b = ?, option_c = ?, option_d = ?, correct_answer = ? WHERE id = ?");
    if ($stmt->execute([$question_text, $question_type, $option_a, $option_b, $option_c, $option_d, $correct_answer, $question_id])) {
        $success = "Question updated successfully!";
        $action = 'view_questions';
    } else {
        $error = "Failed to update question.";
    }
}

// Handle bulk upload
if ($_POST['action'] ?? '' === 'bulk_upload' && isset($_FILES['questions_file'])) {
    $exam_id = intval($_POST['exam_id']);
    $question_type = Config::sanitize($_POST['upload_question_type']);
    
    if ($_FILES['questions_file']['error'] === UPLOAD_ERR_OK) {
        $upload_result = handleBulkUpload($exam_id, $question_type, $_FILES['questions_file']);
        if ($upload_result['success']) {
            $success = "Successfully uploaded " . $upload_result['count'] . " questions!";
            $action = 'view_questions';
        } else {
            $error = "Upload failed: " . $upload_result['error'];
        }
    } else {
        $error = "File upload error: " . $_FILES['questions_file']['error'];
    }
}

// Get exams list
$stmt = $db->prepare("SELECT * FROM exams ORDER BY created_at DESC");
$stmt->execute();
$exams = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get specific exam details
$exam = null;
if ($exam_id) {
    $stmt = $db->prepare("SELECT * FROM exams WHERE id = ?");
    $stmt->execute([$exam_id]);
    $exam = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get questions for exam
$questions = [];
if ($exam_id && in_array($action, ['view_questions', 'edit_question'])) {
    $stmt = $db->prepare("SELECT * FROM questions WHERE exam_id = ? ORDER BY id");
    $stmt->execute([$exam_id]);
    $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get specific question for editing
$question = null;
if ($question_id) {
    $stmt = $db->prepare("SELECT * FROM questions WHERE id = ?");
    $stmt->execute([$question_id]);
    $question = $stmt->fetch(PDO::FETCH_ASSOC);
}

function handleBulkUpload($exam_id, $question_type, $file) {
    $db = Config::getDB();
    $allowed_types = ['text/csv', 'application/vnd.ms-excel', 'text/plain'];
    
    if (!in_array($file['type'], $allowed_types)) {
        return ['success' => false, 'error' => 'Invalid file type. Please upload CSV or Excel file.'];
    }
    
    $handle = fopen($file['tmp_name'], 'r');
    if (!$handle) {
        return ['success' => false, 'error' => 'Cannot open uploaded file.'];
    }
    
    $count = 0;
    $errors = [];
    $first_row = true;
    
    while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
        if ($first_row) {
            $first_row = false;
            continue; // Skip header row
        }
        
        if (count($data) < 3) continue; // Skip invalid rows
        
        $question_text = Config::sanitize($data[1] ?? '');
        if (empty($question_text)) continue;
        
        if ($question_type === 'multiple_choice') {
            $option_a = Config::sanitize($data[2] ?? '');
            $option_b = Config::sanitize($data[3] ?? '');
            $option_c = Config::sanitize($data[4] ?? '');
            $option_d = Config::sanitize($data[5] ?? '');
            $correct_answer = Config::sanitize($data[6] ?? '');
            
            if (empty($option_a) || empty($correct_answer)) {
                $errors[] = "Row " . ($count + 1) . ": Missing required fields";
                continue;
            }
        } else {
            $option_a = $option_b = $option_c = $option_d = null;
            $correct_answer = Config::sanitize($data[2] ?? '');
        }
        
        $stmt = $db->prepare("INSERT INTO questions (exam_id, question_text, question_type, option_a, option_b, option_c, option_d, correct_answer) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$exam_id, $question_text, $question_type, $option_a, $option_b, $option_c, $option_d, $correct_answer])) {
            $count++;
        } else {
            $errors[] = "Row " . ($count + 1) . ": Database insertion failed";
        }
    }
    
    fclose($handle);
    
    if (!empty($errors)) {
        return ['success' => false, 'error' => implode(', ', array_slice($errors, 0, 5))];
    }
    
    return ['success' => true, 'count' => $count];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Exams - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">CBT System - Admin</a>
            <div class="navbar-nav ms-auto">
                <a href="admin_dashboard.php" class="nav-link">Dashboard</a>
                <a href="admin_results.php" class="nav-link">Results</a>
                <a href="logout.php" class="nav-link">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Notifications -->
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>

        <div class="row">
            <div class="col-12">
                <!-- Navigation Tabs -->
                <ul class="nav nav-tabs mb-4">
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'list' ? 'active' : '' ?>" href="?action=list">Exams</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?= in_array($action, ['add_exam', 'edit_exam']) ? 'active' : '' ?>" 
                           href="?action=add_exam"><?= $exam_id ? 'Edit' : 'Add' ?> Exam</a>
                    </li>
                    <?php if ($exam_id && in_array($action, ['view_questions', 'add_question', 'edit_question', 'bulk_upload'])): ?>
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'view_questions' ? 'active' : '' ?>" 
                           href="?action=view_questions&exam_id=<?= $exam_id ?>">View Questions</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'add_question' ? 'active' : '' ?>" 
                           href="?action=add_question&exam_id=<?= $exam_id ?>">Add Question</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link <?= $action === 'bulk_upload' ? 'active' : '' ?>" 
                           href="?action=bulk_upload&exam_id=<?= $exam_id ?>">Bulk Upload</a>
                    </li>
                    <?php endif; ?>
                </ul>

                <!-- Content Sections -->
                <?php if ($action === 'list'): ?>
                    <!-- Exams List -->
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h4 class="mb-0"><i class="fas fa-book me-2"></i>Exams</h4>
                            <a href="?action=add_exam" class="btn btn-primary btn-sm">
                                <i class="fas fa-plus me-1"></i>Add New Exam
                            </a>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-striped" id="examsTable">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Exam Name</th>
                                            <th>Questions</th>
                                            <th>Duration</th>
                                            <th>Points/Correct</th>
                                            <th>Status</th>
                                            <th>Created</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($exams as $exam): ?>
                                        <tr>
                                            <td><?= $exam['id'] ?></td>
                                            <td><?= htmlspecialchars($exam['name']) ?></td>
                                            <td><?= $exam['total_questions'] ?></td>
                                            <td><?= floor($exam['duration_seconds'] / 60) ?> mins</td>
                                            <td><?= $exam['points_per_correct'] ?></td>
                                            <td>
                                                <span class="badge bg-<?= $exam['is_active'] ? 'success' : 'secondary' ?>">
                                                    <?= $exam['is_active'] ? 'Active' : 'Inactive' ?>
                                                </span>
                                            </td>
                                            <td><?= date('M j, Y', strtotime($exam['created_at'])) ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="?action=view_questions&exam_id=<?= $exam['id'] ?>" 
                                                       class="btn btn-info" title="View Questions">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="?action=edit_exam&exam_id=<?= $exam['id'] ?>" 
                                                       class="btn btn-warning" title="Edit Exam">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="?action=bulk_upload&exam_id=<?= $exam['id'] ?>" 
                                                       class="btn btn-success" title="Bulk Upload">
                                                        <i class="fas fa-upload"></i>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                <?php elseif (in_array($action, ['add_exam', 'edit_exam'])): ?>
                    <!-- Add/Edit Exam Form -->
                    <div class="card">
                        <div class="card-header">
                            <h4 class="mb-0">
                                <i class="fas fa-<?= $exam_id ? 'edit' : 'plus' ?> me-2"></i>
                                <?= $exam_id ? 'Edit Exam' : 'Add New Exam' ?>
                            </h4>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="<?= $exam_id ? 'edit_exam' : 'add_exam' ?>">
                                <?php if ($exam_id): ?>
                                    <input type="hidden" name="exam_id" value="<?= $exam_id ?>">
                                <?php endif; ?>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Exam Name</label>
                                            <input type="text" name="name" class="form-control" 
                                                   value="<?= $exam['name'] ?? '' ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="mb-3">
                                            <label class="form-label">Total Questions</label>
                                            <input type="number" name="total_questions" class="form-control" 
                                                   value="<?= $exam['total_questions'] ?? 10 ?>" min="1" required>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="mb-3">
                                            <label class="form-label">Duration (minutes)</label>
                                            <input type="number" name="duration_seconds" class="form-control" 
                                                   value="<?= isset($exam['duration_seconds']) ? $exam['duration_seconds'] / 60 : 30 ?>" min="1" required>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Points per Correct Answer</label>
                                            <input type="number" step="0.1" name="points_per_correct" class="form-control" 
                                                   value="<?= $exam['points_per_correct'] ?? 1.0 ?>" min="0.1" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label class="form-label">Status</label>
                                            <div class="form-check form-switch mt-2">
                                                <input class="form-check-input" type="checkbox" name="is_active" 
                                                       <?= isset($exam['is_active']) && $exam['is_active'] ? 'checked' : 'checked' ?>>
                                                <label class="form-check-label">Active</label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-save me-1"></i>
                                        <?= $exam_id ? 'Update Exam' : 'Create Exam' ?>
                                    </button>
                                    <a href="?action=list" class="btn btn-secondary">Cancel</a>
                                </div>
                            </form>
                        </div>
                    </div>

                <?php elseif ($action === 'view_questions' && $exam_id): ?>
                    <!-- View Questions -->
                    <div class="card">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h4 class="mb-0">
                                <i class="fas fa-list me-2"></i>
                                Questions for: <?= htmlspecialchars($exam['name']) ?>
                            </h4>
                            <div>
                                <a href="?action=add_question&exam_id=<?= $exam_id ?>" class="btn btn-primary btn-sm">
                                    <i class="fas fa-plus me-1"></i>Add Question
                                </a>
                                <a href="?action=bulk_upload&exam_id=<?= $exam_id ?>" class="btn btn-success btn-sm">
                                    <i class="fas fa-upload me-1"></i>Bulk Upload
                                </a>
                            </div>
                        </div>
                        <div class="card-body">
                            <?php if (empty($questions)): ?>
                                <div class="alert alert-info text-center">
                                    <i class="fas fa-info-circle me-2"></i>
                                    No questions found for this exam.
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-striped" id="questionsTable">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Question</th>
                                                <th>Type</th>
                                                <th>Options</th>
                                                <th>Correct Answer</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($questions as $q): ?>
                                            <tr>
                                                <td><?= $q['id'] ?></td>
                                                <td><?= htmlspecialchars(substr($q['question_text'], 0, 100)) ?><?= strlen($q['question_text']) > 100 ? '...' : '' ?></td>
                                                <td>
                                                    <span class="badge bg-<?= $q['question_type'] === 'multiple_choice' ? 'primary' : 'info' ?>">
                                                        <?= ucfirst(str_replace('_', ' ', $q['question_type'])) ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <?php if ($q['question_type'] === 'multiple_choice'): ?>
                                                        A: <?= htmlspecialchars(substr($q['option_a'], 0, 20)) ?>...<br>
                                                        B: <?= htmlspecialchars(substr($q['option_b'], 0, 20)) ?>...
                                                    <?php else: ?>
                                                        <em>Subjective</em>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <code><?= htmlspecialchars($q['correct_answer']) ?></code>
                                                </td>
                                                <td>
                                                    <div class="btn-group btn-group-sm">
                                                        <a href="?action=edit_question&exam_id=<?= $exam_id ?>&question_id=<?= $q['id'] ?>" 
                                                           class="btn btn-warning" title="Edit">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        <button class="btn btn-danger" title="Delete" 
                                                                onclick="confirmDelete(<?= $q['id'] ?>)">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                <?php elseif (in_array($action, ['add_question', 'edit_question']) && $exam_id): ?>
                    <!-- Add/Edit Question Form -->
                    <div class="card">
                        <div class="card-header">
                            <h4 class="mb-0">
                                <i class="fas fa-<?= $question_id ? 'edit' : 'plus' ?> me-2"></i>
                                <?= $question_id ? 'Edit Question' : 'Add New Question' ?>
                                <small class="text-muted">- <?= htmlspecialchars($exam['name']) ?></small>
                            </h4>
                        </div>
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="<?= $question_id ? 'edit_question' : 'add_question' ?>">
                                <input type="hidden" name="exam_id" value="<?= $exam_id ?>">
                                <?php if ($question_id): ?>
                                    <input type="hidden" name="question_id" value="<?= $question_id ?>">
                                <?php endif; ?>
                                
                                <div class="mb-3">
                                    <label class="form-label">Question Type</label>
                                    <select name="question_type" class="form-select" id="questionType" required>
                                        <option value="multiple_choice" <?= ($question['question_type'] ?? '') === 'multiple_choice' ? 'selected' : '' ?>>Multiple Choice</option>
                                        <option value="subjective" <?= ($question['question_type'] ?? '') === 'subjective' ? 'selected' : '' ?>>Subjective</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Question Text</label>
                                    <textarea name="question_text" class="form-control" rows="3" required><?= $question['question_text'] ?? '' ?></textarea>
                                </div>
                                
                                <!-- Multiple Choice Options -->
                                <div id="multipleChoiceOptions" style="<?= ($question['question_type'] ?? 'multiple_choice') === 'subjective' ? 'display: none;' : '' ?>">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Option A</label>
                                                <input type="text" name="option_a" class="form-control" value="<?= $question['option_a'] ?? '' ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Option B</label>
                                                <input type="text" name="option_b" class="form-control" value="<?= $question['option_b'] ?? '' ?>">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Option C</label>
                                                <input type="text" name="option_c" class="form-control" value="<?= $question['option_c'] ?? '' ?>">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label">Option D</label>
                                                <input type="text" name="option_d" class="form-control" value="<?= $question['option_d'] ?? '' ?>">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Correct Answer</label>
                                    <input type="text" name="correct_answer" class="form-control" value="<?= $question['correct_answer'] ?? '' ?>" required>
                                    <small class="form-text text-muted">
                                        <?php if (($question['question_type'] ?? 'multiple_choice') === 'multiple_choice'): ?>
                                            Enter A, B, C, or D
                                        <?php else: ?>
                                            For subjective questions, you can enter comma-separated acceptable answers
                                        <?php endif; ?>
                                    </small>
                                </div>
                                
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-save me-1"></i>
                                        <?= $question_id ? 'Update Question' : 'Add Question' ?>
                                    </button>
                                    <a href="?action=view_questions&exam_id=<?= $exam_id ?>" class="btn btn-secondary">Cancel</a>
                                </div>
                            </form>
                        </div>
                    </div>

                <?php elseif ($action === 'bulk_upload' && $exam_id): ?>
                    <!-- Bulk Upload -->
                    <div class="card">
                        <div class="card-header">
                            <h4 class="mb-0">
                                <i class="fas fa-upload me-2"></i>
                                Bulk Upload Questions
                                <small class="text-muted">- <?= htmlspecialchars($exam['name']) ?></small>
                            </h4>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <form method="POST" enctype="multipart/form-data">
                                        <input type="hidden" name="action" value="bulk_upload">
                                        <input type="hidden" name="exam_id" value="<?= $exam_id ?>">
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Question Type</label>
                                            <select name="upload_question_type" class="form-select" required>
                                                <option value="multiple_choice">Multiple Choice</option>
                                                <option value="subjective">Subjective</option>
                                            </select>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Upload CSV/Excel File</label>
                                            <input type="file" name="questions_file" class="form-control" accept=".csv,.xlsx,.xls" required>
                                            <small class="form-text text-muted">
                                                File must be in CSV format with appropriate columns
                                            </small>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-success">
                                            <i class="fas fa-upload me-1"></i>Upload Questions
                                        </button>
                                        <a href="?action=view_questions&exam_id=<?= $exam_id ?>" class="btn btn-secondary">Cancel</a>
                                    </form>
                                </div>
                                <div class="col-md-6">
                                    <div class="alert alert-info">
                                        <h6><i class="fas fa-info-circle me-2"></i>File Format Requirements</h6>
                                        
                                        <strong>For Multiple Choice Questions:</strong>
                                        <table class="table table-sm mt-2">
                                            <thead>
                                                <tr>
                                                    <th>S/N</th>
                                                    <th>Question</th>
                                                    <th>Option A</th>
                                                    <th>Option B</th>
                                                    <th>Option C</th>
                                                    <th>Option D</th>
                                                    <th>Answer</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <tr>
                                                    <td>1</td>
                                                    <td>What is 2+2?</td>
                                                    <td>3</td>
                                                    <td>4</td>
                                                    <td>5</td>
                                                    <td>6</td>
                                                    <td>B</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                        
                                        <strong>For Subjective Questions:</strong>
                                        <table class="table table-sm mt-2">
                                            <thead>
                                                <tr>
                                                    <th>S/N</th>
                                                    <th>Question</th>
                                                    <th>Answer</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <tr>
                                                    <td>1</td>
                                                    <td>Capital of France?</td>
                                                    <td>Paris</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                        
                                        <p class="mb-0 mt-2">
                                            <strong>Note:</strong> The first row should be headers. Files should be saved as CSV format.
                                        </p>
                                    </div>
                                    
                                    <div class="text-center">
                                        <a href="sample_questions.csv" class="btn btn-outline-primary btn-sm" download>
                                            <i class="fas fa-download me-1"></i>Download Sample CSV
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('#examsTable, #questionsTable').DataTable({
                pageLength: 25,
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>'
            });

            // Show/hide multiple choice options based on question type
            $('#questionType').change(function() {
                if ($(this).val() === 'multiple_choice') {
                    $('#multipleChoiceOptions').show();
                    $('input[name="correct_answer"]').attr('placeholder', 'Enter A, B, C, or D');
                } else {
                    $('#multipleChoiceOptions').hide();
                    $('input[name="correct_answer"]').attr('placeholder', 'Enter correct answer (comma-separated for multiple)');
                }
            });
        });

        function confirmDelete(questionId) {
            if (confirm('Are you sure you want to delete this question?')) {
                // In a real implementation, you would make an AJAX call here
                alert('Delete functionality would be implemented here for question ID: ' + questionId);
            }
        }
    </script>
</body>
</html>