<?php
require_once 'config.php';
require_once 'auth.php';

class ExamManager {
    public static function startExam($user_id, $exam_id) {
        $db = Config::getDB();
        
        // ALWAYS check answers table first - this is the source of truth
        $stmt = $db->prepare("SELECT * FROM answers WHERE user_id = ? AND exam_id = ?");
        $stmt->execute([$user_id, $exam_id]);
        $existing_answer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing_answer) {
            // Get the session data
            $stmt = $db->prepare("SELECT * FROM exam_sessions WHERE id = ?");
            $stmt->execute([$existing_answer['session_id']]);
            $session = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$session) {
                // Session doesn't exist, create a new one
                $session = self::createNewSession($user_id, $exam_id, $existing_answer['time_left_seconds']);
                // Update answers record with new session_id
                $stmt = $db->prepare("UPDATE answers SET session_id = ? WHERE user_id = ? AND exam_id = ?");
                $stmt->execute([$session['id'], $user_id, $exam_id]);
            }
            
            // Always return the existing data - this ensures continuity
            return array_merge($session, [
                'answers_data' => $existing_answer,
                'time_left_seconds' => $existing_answer['time_left_seconds'],
                'question_ids' => json_decode($existing_answer['question_ids'], true),
                'resumed' => true
            ]);
        }
        
        // No existing record - create new one
        return self::createNewExamAttempt($user_id, $exam_id);
    }
    
    private static function createNewSession($user_id, $exam_id, $time_left) {
        $db = Config::getDB();
        
        $start_time = date('Y-m-d H:i:s');
        $stmt = $db->prepare("INSERT INTO exam_sessions (user_id, exam_id, start_time, time_left_seconds) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_id, $exam_id, $start_time, $time_left]);
        $session_id = $db->lastInsertId();
        
        return [
            'id' => $session_id,
            'user_id' => $user_id,
            'exam_id' => $exam_id,
            'start_time' => $start_time,
            'time_left_seconds' => $time_left,
            'is_completed' => false
        ];
    }
    
    private static function createNewExamAttempt($user_id, $exam_id) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT * FROM exams WHERE id = ?");
        $stmt->execute([$exam_id]);
        $exam = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$exam) {
            throw new Exception("Exam not found");
        }
        
        $start_time = date('Y-m-d H:i:s');
        $time_left = $exam['duration_seconds'];
        
        // Create new session
        $stmt = $db->prepare("INSERT INTO exam_sessions (user_id, exam_id, start_time, time_left_seconds) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_id, $exam_id, $start_time, $time_left]);
        $session_id = $db->lastInsertId();
        
        $questions = self::getQuestions($exam_id, $exam['total_questions']);
        if (empty($questions)) {
            throw new Exception("No questions available for exam ID: " . $exam_id);
        }
        
        $question_ids = array_column($questions, 'id');
        $user_answers = array_fill(0, count($question_ids), '');
        
        // Create new answers record
        $stmt = $db->prepare("INSERT INTO answers (session_id, exam_id, user_id, question_ids, user_answers, time_spent, time_left_seconds) 
                             VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $session_id, 
            $exam_id,
            $user_id,
            json_encode($question_ids), 
            json_encode($user_answers),
            json_encode(array_fill(0, count($question_ids), 0)),
            $time_left
        ]);
        
        // Get the final answers data
        $stmt = $db->prepare("SELECT * FROM answers WHERE user_id = ? AND exam_id = ?");
        $stmt->execute([$user_id, $exam_id]);
        $answers_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return [
            'id' => $session_id,
            'user_id' => $user_id,
            'exam_id' => $exam_id,
            'start_time' => $start_time,
            'time_left_seconds' => $time_left,
            'question_ids' => $question_ids,
            'answers_data' => $answers_data
        ];
    }
    
    public static function getQuestions($exam_id, $limit) {
        $db = Config::getDB();
        $limit = max(1, intval($limit));
        
        $stmt = $db->prepare("SELECT * FROM questions WHERE exam_id = ? ORDER BY RAND() LIMIT ?");
        $stmt->bindValue(1, $exam_id, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public static function getQuestionsByIds($question_ids) {
        $db = Config::getDB();
        
        if (empty($question_ids)) return [];
        
        $placeholders = str_repeat('?,', count($question_ids) - 1) . '?';
        $stmt = $db->prepare("SELECT * FROM questions WHERE id IN ($placeholders)");
        $stmt->execute($question_ids);
        
        $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $ordered_questions = [];
        foreach ($question_ids as $id) {
            foreach ($questions as $question) {
                if ($question['id'] == $id) {
                    $ordered_questions[] = $question;
                    break;
                }
            }
        }
        
        return $ordered_questions;
    }
    
    public static function saveAnswer($session_id, $question_index, $answer, $time_spent, $time_left) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT * FROM answers WHERE session_id = ?");
        $stmt->execute([$session_id]);
        $answers_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$answers_data) return false;
        
        $question_ids = json_decode($answers_data['question_ids'], true);
        $user_answers = json_decode($answers_data['user_answers'], true);
        $time_spent_array = json_decode($answers_data['time_spent'], true);
        $is_correct_array = json_decode($answers_data['is_correct'] ?? '[]', true);
        
        // Remove spaces from subjective answers (force single word)
        $cleaned_answer = $answer;
        $question_id = $question_ids[$question_index];
        
        // Check if this is a subjective question
        $stmt = $db->prepare("SELECT question_type FROM questions WHERE id = ?");
        $stmt->execute([$question_id]);
        $question_type = $stmt->fetch(PDO::FETCH_ASSOC)['question_type'];
        
        if ($question_type === 'subjective') {
            $cleaned_answer = str_replace(' ', '', trim($answer)); // Remove all spaces
        }
        
        $user_answers[$question_index] = $cleaned_answer;
        $time_spent_array[$question_index] = $time_spent;
        
        if ($cleaned_answer !== '') {
            $stmt = $db->prepare("SELECT correct_answer, question_type FROM questions WHERE id = ?");
            $stmt->execute([$question_id]);
            $question = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($question) {
                $is_correct = false;
                
                if ($question['question_type'] === 'subjective') {
                    // For subjective questions: check against comma-separated possible answers
                    $correct_answers = explode(',', $question['correct_answer']);
                    $user_clean = strtolower($cleaned_answer);
                    
                    foreach ($correct_answers as $correct_answer) {
                        $correct_clean = strtolower(str_replace(' ', '', trim($correct_answer)));
                        if ($user_clean === $correct_clean) {
                            $is_correct = true;
                            break;
                        }
                    }
                } else {
                    // Exact match for multiple choice
                    $is_correct = ($cleaned_answer === $question['correct_answer']);
                }
                
                if (count($is_correct_array) !== count($question_ids)) {
                    $is_correct_array = array_fill(0, count($question_ids), null);
                }
                $is_correct_array[$question_index] = $is_correct;
            }
        }
        
        $stmt = $db->prepare("UPDATE answers SET 
                            user_answers = ?, 
                            time_spent = ?, 
                            is_correct = ?,
                            current_question_index = ?,
                            time_left_seconds = ?,
                            last_updated = NOW() 
                            WHERE session_id = ?");
        
        return $stmt->execute([
            json_encode($user_answers),
            json_encode($time_spent_array),
            json_encode($is_correct_array),
            $question_index,
            $time_left,
            $session_id
        ]);
    }
    
    public static function submitExam($session_id) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT a.*, e.points_per_correct 
                            FROM answers a 
                            JOIN exam_sessions es ON a.session_id = es.id 
                            JOIN exams e ON es.exam_id = e.id 
                            WHERE a.session_id = ?");
        $stmt->execute([$session_id]);
        $answers_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$answers_data) return false;
        
        $question_ids = json_decode($answers_data['question_ids'], true);
        $user_answers = json_decode($answers_data['user_answers'], true);
        $is_correct_array = [];
        $points_per_correct = $answers_data['points_per_correct'];
        
        $total_score = 0;
        
        foreach ($question_ids as $index => $question_id) {
            $user_answer = $user_answers[$index] ?? '';
            
            if ($user_answer !== '') {
                $stmt = $db->prepare("SELECT correct_answer, question_type FROM questions WHERE id = ?");
                $stmt->execute([$question_id]);
                $question = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($question) {
                    $is_correct = false;
                    
                    if ($question['question_type'] === 'subjective') {
                        // Check against comma-separated possible answers
                        $correct_answers = explode(',', $question['correct_answer']);
                        $user_clean = strtolower($user_answer);
                        
                        foreach ($correct_answers as $correct_answer) {
                            $correct_clean = strtolower(str_replace(' ', '', trim($correct_answer)));
                            if ($user_clean === $correct_clean) {
                                $is_correct = true;
                                break;
                            }
                        }
                    } else {
                        // Exact match for multiple choice
                        $is_correct = ($user_answer === $question['correct_answer']);
                    }
                    
                    $is_correct_array[$index] = $is_correct;
                    
                    if ($is_correct) {
                        $total_score += $points_per_correct;
                    }
                }
            } else {
                $is_correct_array[$index] = false;
            }
        }
        
        // Update the is_correct array
        $stmt = $db->prepare("UPDATE answers SET is_correct = ? WHERE session_id = ?");
        $stmt->execute([json_encode($is_correct_array), $session_id]);
        
        $end_time = date('Y-m-d H:i:s');
        $stmt = $db->prepare("UPDATE exam_sessions SET is_completed = TRUE, end_time = ?, total_score = ? WHERE id = ?");
        return $stmt->execute([$end_time, $total_score, $session_id]);
    }
    
    public static function hasActiveAttempt($user_id, $exam_id) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT a.*, es.is_completed 
                            FROM answers a 
                            JOIN exam_sessions es ON a.session_id = es.id 
                            WHERE a.user_id = ? AND a.exam_id = ? AND es.is_completed = FALSE");
        $stmt->execute([$user_id, $exam_id]);
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }   

    public static function getUserCourse($id){$db = Config::getDB(); $FAPE = 15000;
        $stmt = $db->prepare("SELECT crs_title,crs_price,reg_cost from courses where id =?  ");
        $stmt->execute([$id]);
$row =  $stmt->fetch(PDO::FETCH_ASSOC); $amount = $row['crs_price'];
return $row['crs_title'];
    }

    public static function getUserCourseCost($id){$db = Config::getDB(); $FAPE = 10000;
        $stmt = $db->prepare("SELECT crs_title,crs_price,reg_cost from courses where id =?  ");
        $stmt->execute([$id]);
$row =  $stmt->fetch(PDO::FETCH_ASSOC); $amount = $row['crs_price'];
return $amount;
    }




    public static function getRegFape($id){$db = Config::getDB(); $FAPE = 10000;
        $stmt = $db->prepare("SELECT reg_cost from courses where id =?  ");
        $stmt->execute([$id]);
$row =  $stmt->fetch(PDO::FETCH_ASSOC); $amount = $row['reg_cost']+$FAPE;
return $amount;
    }

    public static function courseEnrolled($id){$db = Config::getDB(); $FAPE = 10000;
        $stmt = $db->prepare("SELECT course_sel from users where id =?  ");
        $stmt->execute([$id]);
return  $stmt->fetch(PDO::FETCH_ASSOC)['course_sel']; 
    }


    
    public static function getExamHistory($user_id, $exam_id) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT a.*, es.start_time, es.end_time, es.total_score, es.is_completed 
                            FROM answers a 
                            JOIN exam_sessions es ON a.session_id = es.id 
                            WHERE a.user_id = ? AND a.exam_id = ? 
                            ORDER BY es.start_time DESC");
        $stmt->execute([$user_id, $exam_id]);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public static function getAttemptCount($user_id, $exam_id) {
        $db = Config::getDB();
        
        $stmt = $db->prepare("SELECT COUNT(*) as attempt_count 
                            FROM answers a 
                            JOIN exam_sessions es ON a.session_id = es.id 
                            WHERE a.user_id = ? AND a.exam_id = ?");
        $stmt->execute([$user_id, $exam_id]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['attempt_count'] ?? 0;
    }
}
?>