<?php
require_once 'auth.php';
require_once 'config.php';
require_once 'ExamManager.php';

if (!Auth::isLoggedIn()) {
    header('Location: admin_login.php');
    exit;
}

if (!Auth::isAdmin()) {
    header('Location: dashboard.php');
    exit;
}

$db = Config::getDB();
$selected_exam_id = $_GET['exam_id'] ?? '';

// Get all exams for the filter dropdown
$stmt = $db->prepare("SELECT * FROM exams WHERE is_active = TRUE ORDER BY name");
$stmt->execute();
$exams = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get results based on filter
if ($selected_exam_id) {
    $stmt = $db->prepare("
        SELECT 
            u.username,
            u.email,
            e.name as exam_name,
            e.points_per_correct,
            MAX(es.total_score) as best_score,
            MAX(es.total_score) * e.points_per_correct as calculated_score,
            COUNT(es.id) as attempt_count,
            MAX(es.end_time) as last_attempt
        FROM users u
        JOIN exam_sessions es ON u.id = es.user_id
        JOIN exams e ON es.exam_id = e.id
        WHERE es.is_completed = TRUE 
        AND e.id = ?
        GROUP BY u.id, e.id
        ORDER BY calculated_score DESC, u.username
    ");
    $stmt->execute([$selected_exam_id]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
    $results = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Exam Results - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container">
            <a class="navbar-brand" href="#">CBT System - Admin</a>
            <div class="navbar-nav ms-auto">
                <a href="admin_dashboard.php" class="nav-link">Dashboard</a>
                <a href="logout.php" class="nav-link">Logout</a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Exam Results</h4>
                        <div class="d-flex gap-2">
                            <select id="examFilter" class="form-select" style="width: auto;">
                                <option value="">Select Exam</option>
                                <?php foreach ($exams as $exam): ?>
                                    <option value="<?= $exam['id'] ?>" <?= $selected_exam_id == $exam['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($exam['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <?php if ($selected_exam_id): ?>
                                <button id="exportBtn" class="btn btn-success btn-sm">
                                    <i class="fas fa-download me-1"></i>Export CSV
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if ($selected_exam_id): ?>
                            <?php if (!empty($results)): ?>
                                <div class="table-responsive">
                                    <table id="resultsTable" class="table table-striped table-bordered">
                                        <thead class="table-dark">
                                            <tr>
                                                <th>S/N</th>
                                                <th>Username</th>
                                                <th>Email</th>
                                                <th>Paper</th>
                                                <th>Attempts</th>
                                                <th>Best Score</th>
                                                <th>Points per Correct</th>
                                                <th>Total Score</th>
                                                <th>Last Attempt</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($results as $index => $result): ?>
                                            <tr>
                                                <td><?= $index + 1 ?></td>
                                                <td><?= htmlspecialchars($result['username']) ?></td>
                                                <td><?= htmlspecialchars($result['email']) ?></td>
                                                <td><?= htmlspecialchars($result['exam_name']) ?></td>
                                                <td class="text-center"><?= $result['attempt_count'] ?></td>
                                                <td class="text-center"><?= number_format($result['best_score'], 2) ?></td>
                                                <td class="text-center"><?= number_format($result['points_per_correct'], 2) ?></td>
                                                <td class="text-center fw-bold text-primary">
                                                     <?php $sumScore = 30 + $result['best_score'];   ?>
                                                    <?= number_format($sumScore, 2) ?>
                                                </td>
                                                <td><?= date('M j, Y g:i A', strtotime($result['last_attempt'])) ?></td>
                                                <td class="text-center">
                                                    <button class="btn btn-info btn-sm view-details" 
                                                            data-username="<?= htmlspecialchars($result['username']) ?>"
                                                            data-exam="<?= htmlspecialchars($result['exam_name']) ?>"
                                                            title="View Attempt Details">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                        <tfoot>
                                            <tr class="table-info">
                                                <td colspan="5" class="text-end fw-bold">Summary:</td>
                                                <td class="text-center fw-bold"><?= number_format(array_sum(array_column($results, 'best_score')) / count($results), 2) ?> (Avg)</td>
                                                <td></td>
                                                <td class="text-center fw-bold"><?= number_format(array_sum(array_column($results, 'calculated_score')), 2) ?> (Total)</td>
                                                <td colspan="2" class="text-center fw-bold"><?= count($results) ?> Students</td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-info text-center">
                                    <i class="fas fa-info-circle me-2"></i>
                                    No results found for the selected exam.
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="text-center py-5">
                                <i class="fas fa-filter fa-3x text-muted mb-3"></i>
                                <h5 class="text-muted">Select an exam to view results</h5>
                                <p class="text-muted">Choose an exam from the dropdown above to see student results.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal for attempt details -->
    <div class="modal fade" id="detailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Attempt Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="modalBody">
                    <!-- Details will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTable
            $('#resultsTable').DataTable({
                pageLength: 25,
                order: [[7, 'desc']], // Sort by Total Score descending
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>'
            });

            // Exam filter change
            $('#examFilter').change(function() {
                const examId = $(this).val();
                if (examId) {
                    window.location.href = 'admin_results.php?exam_id=' + examId;
                } else {
                    window.location.href = 'admin_results.php';
                }
            });

            // Export to CSV
            $('#exportBtn').click(function() {
                const examName = $('#examFilter option:selected').text();
                const table = $('#resultsTable').DataTable();
                const data = table.data().toArray();
                
                let csv = 'S/N,Username,Email,Paper,Attempts,Best Score,Points per Correct,Total Score,Last Attempt\n';
                
                data.forEach((row, index) => {
                    csv += `"${index + 1}","${row[1]}","${row[2]}","${row[3]}","${row[4]}","${row[5]}","${row[6]}","${row[7]}","${row[8]}"\n`;
                });
                
                const blob = new Blob([csv], { type: 'text/csv' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `exam_results_${examName.replace(/\s+/g, '_')}_${new Date().toISOString().split('T')[0]}.csv`;
                a.click();
                window.URL.revokeObjectURL(url);
            });

            // View attempt details
            $('.view-details').click(function() {
                const username = $(this).data('username');
                const examName = $(this).data('exam');
                
                $('#modalBody').html(`
                    <div class="text-center">
                        <div class="spinner-border" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p>Loading details...</p>
                    </div>
                `);
                
                $('#detailsModal').modal('show');
                
                // In a real implementation, you would fetch detailed attempt history here
                setTimeout(() => {
                    $('#modalBody').html(`
                        <h6>Student: ${username}</h6>
                        <h6>Exam: ${examName}</h6>
                        <hr>
                        <p class="text-muted">Detailed attempt history would be displayed here.</p>
                        <p>This could include:</p>
                        <ul>
                            <li>Individual attempt scores</li>
                            <li>Time spent on each attempt</li>
                            <li>Question-by-question analysis</li>
                            <li>Progress over time</li>
                        </ul>
                    `);
                }, 1000);
            });
        });
    </script>

    <style>
        .table th {
            font-weight: 600;
        }
        .dataTables_wrapper .dataTables_length select {
            width: auto;
            display: inline-block;
        }
        .view-details {
            transition: all 0.3s ease;
        }
        .view-details:hover {
            transform: scale(1.1);
        }
    </style>
</body>
</html>