<?php
require_once 'auth.php';
require_once 'ExamManager.php';

if (!Auth::isLoggedIn()) {
    header('Location: index.php');
    exit;
}

$exam_id = intval($_GET['exam_id'] ?? 0);
if (!$exam_id) {
    die("Invalid exam");
}

$session = ExamManager::startExam($_SESSION['user_id'], $exam_id);

// Always use the question_ids from answers_data
$question_ids = json_decode($session['answers_data']['question_ids'], true);
$questions = ExamManager::getQuestionsByIds($question_ids);

if (empty($questions)) {
    die("No questions available for this exam. Please contact administrator.");
}

// Get saved answers from the answers_data
$saved_answers = json_decode($session['answers_data']['user_answers'], true);
$time_spent = json_decode($session['answers_data']['time_spent'] ?? '[]', true);
$current_question_index = $session['answers_data']['current_question_index'] ?? 0;

// Use the time from answers_data (this is the stored time)
$time_left = $session['answers_data']['time_left_seconds'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Exam - CBT System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="icon" type="image/png" href="brainpals_logo.png">
    <style>

<style>
    /* Disable text selection */
    body {
      -webkit-user-select: none; /* Safari */
      -moz-user-select: none;    /* Firefox */
      -ms-user-select: none;     /* IE10+/Edge */
      user-select: none;         /* Standard */
    }

    /* Prevent callout menu on iOS Safari */
    body {
      -webkit-touch-callout: none;
    }
    .question-indicator {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        font-weight: bold;
        transition: all 0.3s ease;
        border: 2px solid;
    }
    
    .question-indicator:hover {
        transform: scale(1.1);
    }
    
    .question-indicator.btn-success {
        border-color: #198754;
    }
    
    .question-indicator.btn-primary {
        border-color: #0d6efd;
    }
    
    .question-indicator.btn-outline-secondary {
        border-color: #6c757d;
    }
    
    #question-status {
        max-width: 100%;
        overflow-x: auto;
        white-space: nowrap;
        padding: 5px 0;
    }
    
    /* Keyboard shortcut hint */
    .keyboard-hint {
        font-size: 0.8rem;
        color: #6c757d;
    }
</style>
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container">
            <span class="navbar-brand">Exam in Progress</span>
            <div class="navbar-text text-white" style="font-size:25px;">
                Time Left: <span id="timer" ><?= $time_left ?></span>s
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="card">
        <div class="card-header">
    <div class="d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Question Navigation</h5>
        <div id="question-status" class="small">
            <?php for ($i = 0; $i < count($questions); $i++): ?>
                <button type="button" 
                        class="btn btn-sm m-1 question-indicator <?= $i == $current_question_index ? 'btn-primary' : (!empty($saved_answers[$i]) ? 'btn-success' : 'btn-outline-secondary') ?>" 
                        data-question="<?= $i ?>"
                        onclick="navigateToQuestion(<?= $i ?>)">
                    <?= $i + 1 ?>
                </button>
            <?php endfor; ?>
        </div>
    </div>
</div>

<!-- <div class="text-center keyboard-hint mt-2">
    <small>
        <i class="fas fa-keyboard me-1"></i>
        Keyboard shortcuts: 
        <kbd>←</kbd> or <kbd>A</kbd> for Previous, 
        <kbd>→</kbd> or <kbd>D</kbd> for Next, 
        <kbd>1-9</kbd> to jump to questions
    </small>
</div> -->

            <div class="card-body">
                <div id="question-container">
                    <?php foreach ($questions as $index => $question): ?>
                    <div class="question-page" id="question-<?= $index ?>" style="display: <?= $index === $current_question_index ? 'block' : 'none' ?>;">
                        <h5>Question <?= $index + 1 ?></h5>
                        <p><?= htmlspecialchars($question['question_text']) ?></p>
                        
                        <?php if ($question['question_type'] === 'multiple_choice'): ?>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="answer-<?= $index ?>" value="A" id="optA-<?= $index ?>"
                                    <?= ($saved_answers[$index] ?? '') === 'A' ? 'checked' : '' ?>>
                                <label class="form-check-label" for="optA-<?= $index ?>">A. <?= htmlspecialchars($question['option_a']) ?></label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="answer-<?= $index ?>" value="B" id="optB-<?= $index ?>"
                                    <?= ($saved_answers[$index] ?? '') === 'B' ? 'checked' : '' ?>>
                                <label class="form-check-label" for="optB-<?= $index ?>">B. <?= htmlspecialchars($question['option_b']) ?></label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="answer-<?= $index ?>" value="C" id="optC-<?= $index ?>"
                                    <?= ($saved_answers[$index] ?? '') === 'C' ? 'checked' : '' ?>>
                                <label class="form-check-label" for="optC-<?= $index ?>">C. <?= htmlspecialchars($question['option_c']) ?></label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="answer-<?= $index ?>" value="D" id="optD-<?= $index ?>"
                                    <?= ($saved_answers[$index] ?? '') === 'D' ? 'checked' : '' ?>>
                                <label class="form-check-label" for="optD-<?= $index ?>">D. <?= htmlspecialchars($question['option_d']) ?></label>
                            </div>
                        <?php else: ?>
                            <div class="form-group">
                                <input type="text" class="form-control" name="answer-<?= $index ?>" 
                                       placeholder="Enter your answer" value="<?= htmlspecialchars($saved_answers[$index] ?? '') ?>">
                            </div>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>

                <div class="mt-4">
                    <button class="btn btn-secondary" id="prev-btn" <?= $current_question_index === 0 ? 'disabled' : '' ?>>Previous</button>
                    <button class="btn btn-primary" id="next-btn"><?= $current_question_index === count($questions) - 1 ? 'Finish' : 'Next' ?></button>
                    <button class="btn btn-success float-end" id="submit-btn">Submit Exam</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
    const SESSION_ID = <?= $session['id'] ?>;
    const TOTAL_QUESTIONS = <?= count($questions) ?>;
    const AUTOSAVE_INTERVAL = <?= Config::AUTOSAVE_INTERVAL ?> * 1000;
    
    let currentQuestion = <?= $current_question_index ?>;
    let timeLeft = <?= $time_left ?>;
    let timerInterval;
    let timeSpent = <?= json_encode($time_spent) ?>;
    let questionStartTime = Date.now();

    $(document).ready(function() {
        startTimer();
        updateNavigation();
        startQuestionTimer();
        updateQuestionIndicators();

        $('#next-btn').click(function() {
            navigateToQuestion(currentQuestion + 1);
        });

        $('#prev-btn').click(function() {
            navigateToQuestion(currentQuestion - 1);
        });

        $('#submit-btn').click(function() {
            if (confirm('Are you sure you want to submit your exam?')) {
                submitExam();
            }
        });

        // Auto-save when answers change
        $('input[type="radio"]').change(function() {
            saveCurrentAnswer();
        });

        let textInputTimer;
        $('input[type="text"]').on('input', function() {
            clearTimeout(textInputTimer);
            textInputTimer = setTimeout(() => {
                saveCurrentAnswer();
            }, 500);
        });

        // Keyboard navigation
        // $(document).keydown(function(e) {
        //     // Left arrow key or A key for previous
        //     if (e.keyCode === 37 || e.keyCode === 65) {
        //         if (currentQuestion > 0) {
        //             navigateToQuestion(currentQuestion - 1);
        //         }
        //     }
        //     // Right arrow key or D key for next
        //     else if (e.keyCode === 39 || e.keyCode === 68) {
        //         if (currentQuestion < TOTAL_QUESTIONS - 1) {
        //             navigateToQuestion(currentQuestion + 1);
        //         }
        //     }
        //     // Number keys 1-9 for direct navigation
        //     else if (e.keyCode >= 49 && e.keyCode <= 57) {
        //         const questionNumber = e.keyCode - 49; // Convert to 0-based index
        //         if (questionNumber < TOTAL_QUESTIONS) {
        //             navigateToQuestion(questionNumber);
        //         }
        //     }
        // });


    });

    function startTimer() {
        timerInterval = setInterval(function() {
            timeLeft--;
            $('#timer').text(timeLeft);

            if (timeLeft <= 0) {
                clearInterval(timerInterval);
                submitExam();
            }
        }, 1000);
    }

    function startQuestionTimer() {
        questionStartTime = Date.now();
    }

    function getTimeSpentOnCurrentQuestion() {
        return Math.floor((Date.now() - questionStartTime) / 1000);
    }

    // Make this function global so it can be called from onclick
    function navigateToQuestion(newIndex) {
        if (newIndex >= 0 && newIndex < TOTAL_QUESTIONS) {
            // Save current answer and time spent before navigating
            saveCurrentAnswer();
            
            // Update time spent for current question
            timeSpent[currentQuestion] += getTimeSpentOnCurrentQuestion();
            
            // Hide current question, show new question
            $('.question-page').hide();
            $('#question-' + newIndex).show();
            
            currentQuestion = newIndex;
            startQuestionTimer();
            updateNavigation();
            updateQuestionIndicators();
            
            // Scroll to top of question container for better UX
            document.getElementById('question-container').scrollIntoView({ 
                behavior: 'smooth', 
                block: 'start' 
            });
        }
    }

    function updateNavigation() {
        $('#prev-btn').prop('disabled', currentQuestion === 0);
        $('#next-btn').text(currentQuestion === TOTAL_QUESTIONS - 1 ? 'Finish' : 'Next');
        
        // Update button states
        $('#prev-btn').toggleClass('btn-secondary', currentQuestion === 0);
        $('#prev-btn').toggleClass('btn-primary', currentQuestion > 0);
        
        $('#next-btn').toggleClass('btn-primary', currentQuestion < TOTAL_QUESTIONS - 1);
        $('#next-btn').toggleClass('btn-success', currentQuestion === TOTAL_QUESTIONS - 1);
    }

    function updateQuestionIndicators() {
        // Remove all active classes first
        $('.question-indicator')
            .removeClass('btn-primary btn-success btn-outline-secondary')
            .addClass('btn-outline-secondary');
        
        // Highlight current question
        $(`.question-indicator[data-question="${currentQuestion}"]`)
            .removeClass('btn-outline-secondary')
            .addClass('btn-primary');
        
        // Highlight answered questions (except current one)
        for (let i = 0; i < TOTAL_QUESTIONS; i++) {
            if (i !== currentQuestion) {
                const questionElement = $('#question-' + i);
                let answer = '';
                
                if (questionElement.find('input[type="radio"]').length > 0) {
                    answer = questionElement.find('input[type="radio"]:checked').val() || '';
                } else {
                    answer = questionElement.find('input[type="text"]').val() || '';
                }
                
                if (answer !== '') {
                    $(`.question-indicator[data-question="${i}"]`)
                        .removeClass('btn-outline-secondary')
                        .addClass('btn-success');
                }
            }
        }
    }

    function saveCurrentAnswer() {
        const questionElement = $('#question-' + currentQuestion);
        let answer = '';

        if (questionElement.find('input[type="radio"]').length > 0) {
            answer = questionElement.find('input[type="radio"]:checked').val() || '';
        } else {
            answer = questionElement.find('input[type="text"]').val() || '';
        }

        // Update question indicator
        const indicator = $(`.question-indicator[data-question="${currentQuestion}"]`);
        if (answer !== '') {
            indicator.removeClass('btn-outline-secondary btn-primary').addClass('btn-success');
        } else {
            indicator.removeClass('btn-success').addClass('btn-primary');
        }

        // Calculate time spent including current session
        const totalTimeSpent = timeSpent[currentQuestion] + getTimeSpentOnCurrentQuestion();

        $.post('save_answer.php', {
            session_id: SESSION_ID,
            question_index: currentQuestion,
            answer: answer,
            time_spent: totalTimeSpent,
            time_left: timeLeft
        }, function(response) {
            if (!response.success) {
                console.error('Failed to save answer:', response.error);
            }
        }, 'json').fail(function(xhr, status, error) {
            console.error('Save answer failed:', error);
        });
    }

    function setupAutosave() {
        setInterval(function() {
            saveCurrentAnswer();
        }, AUTOSAVE_INTERVAL);
    }

    function submitExam() {
        saveCurrentAnswer();
        
        // Show loading state
        $('#submit-btn').prop('disabled', true).text('Submitting...');
        
        $.post('submit_exam.php', {
            session_id: SESSION_ID
        }, function(response) {
            if (response.success) {
                alert('Exam submitted successfully!');
                window.location.href = 'dashboard.php';
            } else {
                alert('Error submitting exam: ' + (response.error || 'Unknown error'));
                $('#submit-btn').prop('disabled', false).text('Submit Exam');
            }
        }, 'json').fail(function(xhr, status, error) {
            alert('Error submitting exam. Please try again.');
            console.error('Submit exam failed:', error);
            $('#submit-btn').prop('disabled', false).text('Submit Exam');
        });
    }

    // Initialize autosave
    setupAutosave();

    // Prevent spaces in subjective answer inputs
$('input[type="text"]').on('keydown', function(e) {
    if (e.keyCode === 32) { // Space key
        e.preventDefault();
        return false;
    }
});

// Also remove spaces on paste
$('input[type="text"]').on('paste', function(e) {
    e.preventDefault();
    const pastedText = (e.originalEvent.clipboardData || window.clipboardData).getData('text');
    const cleanedText = pastedText.replace(/\s/g, ''); // Remove all spaces
    document.execCommand('insertText', false, cleanedText);
});
</script>

<script>
    // Disable right-click context menu
    document.addEventListener("contextmenu", function (e) {
      e.preventDefault();
    });

    // Disable copy event
    document.addEventListener("copy", function (e) {
      e.preventDefault();
      alert("Copying is disabled on this page.");
    });

    // Disable cut event
    document.addEventListener("cut", function (e) {
      e.preventDefault();
      alert("Cutting is disabled on this page.");
    });

    // Disable paste event
    document.addEventListener("paste", function (e) {
      e.preventDefault();
      alert("Pasting is disabled on this page.");
    });
  </script>
</body>
</html>